<?php

namespace App\Http\Controllers\Api;

use App\Mail\ForgotPassword;
use App\Mail\AccountCreated;
use App\Models\DeliveryAgent;
use App\Models\User;
use GrahamCampbell\ResultType\Success;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Spatie\Permission\Models\Role;
// use Spatie\Permission\Models\Permission;
use Laravel\Passport\TokenRepository;

class AuthApiController extends BaseController
{
    //

    public function login(Request $request)
    {
        
        $request->validate([
            'email' => 'required',
            'password' => 'required', 
        ]);

        $user = User::with('roles','permissions')->where('email',$request->email)->first();

        if ($user == null) {
            $user = DeliveryAgent::with('roles','permissions')
            ->where('email',$request->email)
            ->orWhere('username',$request->email)
            ->first();
            if ($user == null) {
                return response([
                    'success' => 0,
                    'message' => 'Unknown User'
                ]);
            }
        }

        if(!password_verify($request->password, $user->password)){
            return response(['success'=>0,'message'=>'Incorrect password']);
        }
        
        switch ($user->roles[0]->name) {
            case 'SUPER ADMIN':
                $accessToken = $user->createToken('authToken', ['super_admin'])->accessToken;
                break;
            case 'ADMIN':
                $accessToken = $user->createToken('authToken', ['admin'])->accessToken;
                break;
            case 'MANAGER':
                $accessToken = $user->createToken('authToken', ['manager'])->accessToken;
                break;
            case 'INTERNAL DELIVERY AGENT':
                $accessToken = $user->createToken('authToken', ['agent'])->accessToken;
                break;
            case 'FREELANCE DELIVERY AGENT':
                $accessToken = $user->createToken('authToken', ['agent'])->accessToken;
                break;
            
            default:
                # code...
                break;
        }

        return response([
            'success' => 1,
            'user' => $user,
            'access_token' => $accessToken
        ]);
    }

    public function forgotPassword(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'agent' => 'required'
        ]);

        if ($request->agent || $request->agent == "true") {
            $user = DeliveryAgent::where('email',$request->email)->first();
        } else {
            $user = User::where('email',$request->email)->first();
        }
        
        if ($user == null) {
            return response(['success'=>0,'message'=>'Unknown User']);
        }

        $password = generateRandomString();
        $user->password = password_hash($password,PASSWORD_DEFAULT);
        $user->save();

        // mail to user
        if ($request->agent || $request->agent == "true") {
            Mail::to($user->email)->send(new ForgotPassword($password,$user->username));
        } else {
            Mail::to($user->email)->send(new ForgotPassword($password,$user->name));
        }
        
        return response([
            'success' => 1,
            'message' => 'New password has been sent to you via email, please check your email'
        ]);
        
    }


    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required | unique:users,email',
            'role' => 'required',
        ]);

        $user = new User();
        $user->name = $request->name;
        $user->email = $request->email;
        if ($request->img) {
            $user->img = $request->img; 
        }
        if ($request->branch_id) {
            $user->branch_id = $request->branch_id; 
        }
        $password = generateRandomString();
        $user->password = password_hash($password,PASSWORD_DEFAULT);
        $user->save();

        // send password to given mail
        Log::info('Password created for '.$request->name,['password' => $password]);
        // assign role
        $user->assignRole($request->role);
        // 
        Mail::to($user->email)->send(new AccountCreated($password,$user->name));
        // 

        return response([
            'success' => 1,
            'user' => $user,
            'password' => $password
        ]);
    }

    public function logout(Request $request)
    {
        // $user = $request->user();
        // $user->revoke();
        
       $tokenRepository = app(TokenRepository::class);
        // Revoke an access token...
       $tokenRepository->revokeAccessToken($request->token);
        
        return response([
                    'message' => 'You have been logged out succesfully!',
                    'success'  => 1
                ]);
    }

    public function changePassword(Request $request)
    {
        $request->validate([
            'old_password' => 'required|string',
            'password' => 'required|string|confirmed'
        ]);

        // match old password
        if (Hash::check($request->old_password, Auth::user()->password)){

            $user = User::find(auth()->user()->id);
            $user->password = Hash::make($request->password);
            $user->save();

            return response([
                    'message' => 'Password has been changed',
                    'status'  => 1
                    ]);
            
        }
        return response([
                    'message' => 'Password not matched!',
                    'status'  => 0
                ]);
    }

    // public function createRoles()
    // {
    //     Role::create(['guard_name' => 'agent','name' => 'INTERNAL DELIVERY AGENT',]);
    //     Role::create(['guard_name' => 'agent','name' => 'FREELANCE DELIVERY AGENT',]);

    // }

    public function index($role)
    {
        switch ($role) {
            case 'admin':
                $users = User::role('ADMIN')->with('branch')->get();
                break;
            
            case 'super-admin':
                $users = User::role('SUPER_ADMIN')->with('branch')->get();
                break;
            
            case 'manager':
                $users = User::role('MANAGER')->with('branch')->get();
                break;
            
            case 'all':
                $users = User::with('roles','branch')->get();
                break;
            
            default:
                return response([
                    'success' => 0,
                    'message' => 'Unknown Role',
                ]);
                break;
        }

        return response([
            'success' => 1,
            'users' => $users
        ]);
    }

    public function show($id)
    {
        $user = User::with('roles')->where('id',$id)->first();
        if ($user) {
            return response([
                'user' => $user,
                'success' => 1 
            ]);
        }

        return response([
            'success' => 0,
            'message' => 'Unknown User'
        ]);
    }
    
    public function getRoles()
    {
        $roles = Role::all();
        return response([
            'roles' => $roles,
            'success' => 1
        ]);
    }
}
